<?php

/**
 * JCH Optimize - Performs several front-end optimizations for fast downloads
 *
 *  @package   jchoptimize/core
 *  @author    Samuel Marshall <samuel@jch-optimize.net>
 *  @copyright Copyright (c) 2024 Samuel Marshall / JCH Optimize
 *  @license   GNU/GPLv3, or later. See LICENSE file
 *
 *  If LICENSE file missing, see <http://www.gnu.org/licenses/>.
 */

namespace JchOptimize\Core\Service\Provider;

use _JchOptimizeVendor\Joomla\DI\Container;
use _JchOptimizeVendor\Joomla\DI\ServiceProviderInterface;
use _JchOptimizeVendor\Laminas\Log\Exception\InvalidArgumentException;
use _JchOptimizeVendor\Laminas\Log\Exception\RuntimeException;
use _JchOptimizeVendor\Laminas\Log\Logger;
use _JchOptimizeVendor\Laminas\Log\Processor\Backtrace;
use _JchOptimizeVendor\Laminas\Log\PsrLoggerAdapter;
use _JchOptimizeVendor\Laminas\Log\Writer\Stream;
use _JchOptimizeVendor\Laminas\Log\Writer\Syslog;
use _JchOptimizeVendor\Laminas\Log\Writer\WriterInterface;
use _JchOptimizeVendor\Psr\Log\LoggerInterface;
use JchOptimize\Core\Platform\PathsInterface;

use function error_log;
use function fileperms;
use function octdec;
use function sprintf;
use function substr;

use const PHP_EOL;

class PsrLogger implements ServiceProviderInterface
{
    public function register(Container $container): void
    {
        $container->share(LoggerInterface::class, function (Container $container): LoggerInterface {
            $logger = new Logger();
            $logger->addWriter($this->getWriter($container));
            $logger->addProcessor(new Backtrace(['ignoredNamespaces' => [
                'Psr\\Log\\AbstractLogger',
                'JchOptimize\\Core\\Debugger'
                ]]));

            return new PsrLoggerAdapter($logger);
        });
    }

    private function getWriter(Container $container): WriterInterface
    {
        $logsPath = $container->get(PathsInterface::class)->getLogsPath() . '/jch-optimize.log';

        try {
            $filePermission = (int) octdec(substr(sprintf('%o', fileperms(__FILE__)), -4)) ?: 0644;
            $filePermission = $filePermission | 0600;

            return new Stream(
                $logsPath,
                'a',
                PHP_EOL,
                $filePermission
            );
        } catch (InvalidArgumentException | RuntimeException $e) {
            error_log($e->getMessage() . ':using Syslog writer', 3, $logsPath);
            return new Syslog();
        }
    }
}
