<?php

/**
 * JCH Optimize - Performs several front-end optimizations for fast downloads
 *
 * @package   jchoptimize/core
 * @author    Samuel Marshall <samuel@jch-optimize.net>
 * @copyright Copyright (c) 2023 Samuel Marshall / JCH Optimize
 * @license   GNU/GPLv3, or later. See LICENSE file
 *
 *  If LICENSE file missing, see <http://www.gnu.org/licenses/>.
 */

namespace JchOptimize\Core\Uri;

use JchOptimize\Core\Uri\Uri;
use _JchOptimizeVendor\GuzzleHttp\Psr7\UriResolver;
use _JchOptimizeVendor\Psr\Http\Message\UriInterface;
use JchOptimize\Core\Cdn\Cdn;
use JchOptimize\Core\Helper;
use JchOptimize\Core\SystemUri;
use JchOptimize\Core\Platform\PathsInterface;

use function str_replace;

final class UriConverter
{
    public static function uriToFilePath(UriInterface $uri, Cdn $cdn, PathsInterface $pathsUtils): string
    {
        $resolvedUri = UriResolver::resolve(SystemUri::currentUri(), $uri);

        $path = str_replace(
            Utils::originDomains($cdn),
            Helper::appendTrailingSlash($pathsUtils->rootPath()),
            (string)$resolvedUri->withQuery('')->withFragment('')
        );

        //convert all directory to unix style
        return strtr(rawurldecode($path), '\\', '/');
    }

    public static function absToNetworkPathReference(UriInterface $uri): UriInterface
    {
        if (!Uri::isAbsolute($uri)) {
            return $uri;
        }

        if ($uri->getUserInfo() != '') {
            return $uri;
        }

        return $uri->withScheme('')->withHost('')->withPort(null);
    }

    public static function filePathToUri(string|UriInterface $url, Cdn $cdn, PathsInterface $pathsUtils): UriInterface
    {
        $uri = Utils::uriFor($url);
        $baseUri = Utils::uriFor(Helper::appendTrailingSlash($pathsUtils->basePath()));
        $relPath = str_replace((string)$baseUri, '', (string)$uri);

        $uri = UriResolver::resolve(Utils::uriFor(SystemUri::siteBaseFull($pathsUtils)), Utils::uriFor($relPath));

        return $cdn->loadCdnResource($uri);
    }
}
